<?php
/**
 * User Membership
 *
 * @package     AutomatorWP\Integrations\LifterLMS\Actions\User_Membership
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LifterLMS_User_Membership extends AutomatorWP_Integration_Action {

    public $integration = 'lifterlms';
    public $action = 'lifterlms_user_membership';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Enroll or unroll user from a membership', 'automatorwp-pro' ),
            'select_option'     => __( 'Enroll or unroll user from <strong>a membership</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Post title. */
            'edit_label'        => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{post}' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Post title. */
            'log_label'         => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{post}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'enroll'     => __( 'Enroll', 'automatorwp-pro' ),
                                'unroll'    => __( 'Unroll', 'automatorwp-pro' ),
                            ),
                            'default' => 'enroll'
                        ),
                    )
                ),
                'post' => automatorwp_utilities_post_option( array(
                    'name'              => __( 'Membership:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'all memberships', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Membership ID', 'automatorwp-pro' ),
                    'post_type'         => 'llms_membership',
                ) ),
            ),
        ) );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Dynamic edit and log labels
        add_filter( 'automatorwp_parse_automation_item_edit_label', array( $this, 'dynamic_label' ), 10, 5 );
        add_filter( 'automatorwp_parse_automation_item_log_label', array( $this, 'dynamic_label' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Custom edit/log label
     *
     * @since 1.0.0
     *
     * @param string    $label      The edit label
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The item type (trigger|action)
     * @param string    $context    The context this function is executed
     * @param array     $type_args  The type parameters
     *
     * @return string
     */
    public function dynamic_label( $label, $object, $item_type, $context, $type_args ) {

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $label;
        }

        // Get the operation value
        ct_setup_table( "automatorwp_{$item_type}s" );
        $operation = ct_get_object_meta( $object->id, 'operation', true );
        ct_reset_setup_table();

        // Update the edit label
        if( $operation === 'unroll' ) {
            return sprintf( __( '%1$s user from %2$s', 'automatorwp-pro' ), '{operation}', '{post}' );
        }

        return $label;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $operation = $action_options['operation'];
        $membership_id = $action_options['post'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'enroll';
        }

        $memberships = array();

        // Check specific membership
        if( $membership_id !== 'any' ) {

            $membership = get_post( $membership_id );

            // Bail if membership doesn't exists
            if( ! $membership ) {
                return;
            }

            $memberships = array( $membership_id );

        }

        switch ( $operation ) {
            case 'enroll':
                // If enrolling to all memberships, get all memberships
                if( $membership_id === 'any' ) {

                    $query = new WP_Query( array(
                        'post_type'		=> 'llms_membership',
                        'post_status'	=> 'publish',
                        'fields'        => 'ids',
                        'nopaging'      => true,
                    ) );

                    $memberships = $query->get_posts();
                }

                // Enroll user in memberships
                foreach( $memberships as $membership_id ) {
                    llms_enroll_student( $user_id, $membership_id );
                }
                break;
            case 'unroll':
                // If unrolling from all memberships, get all user memberships
                if( $membership_id === 'any' ) {

                    $student = new LLMS_Student( $user_id );
                    $user_memberships = $student->get_memberships( array( 'limit' => 999 ) );

                    // Bail if user is not in any membership
                    if( empty( $user_memberships['results'] ) ) {
                        return;
                    }

                    $memberships = $user_memberships['results'];

                }

                // Unroll user in memberships
                foreach( $memberships as $membership_id ) {
                    llms_unenroll_student( $user_id, $membership_id );
                }
                break;
        }

    }

}

new AutomatorWP_LifterLMS_User_Membership();